VERSION 1.0 CLASS
BEGIN
  MultiUse = -1  'True
END
Attribute VB_Name = "cOrbStream"
Attribute VB_GlobalNameSpace = False
Attribute VB_Creatable = True
Attribute VB_PredeclaredId = False
Attribute VB_Exposed = True
'Copyright (c) 1999 Martin.Both

'This library is free software; you can redistribute it and/or
'modify it under the terms of the GNU Library General Public
'License as published by the Free Software Foundation; either
'version 2 of the License, or (at your option) any later version.

'This library is distributed in the hope that it will be useful,
'but WITHOUT ANY WARRANTY; without even the implied warranty of
'MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
'Library General Public License for more details.

Option Explicit

'Set DebugMode = 0 to deactivate debug code in this class
#Const DebugMode = 0

#If DebugMode Then
    Private lClassDebugID As Long
#End If

'Is initialized?
Private oOrb As cOrbImpl

'Common Data Representation (CDR)

Private buffer() As Byte
Private bufOff As Long  'Offset (= bufStart)
Private bufEof As Long  '(= bufEnd + 1), (bufLen = bufEof - bufStart)
Private bufPos As Long  'Next Position (buffer(bufPos))
Private bLittleEndian As Boolean

'Encapsulation Stack
Private Type tEncap
    bufOff As Long
    bufEof As Long
    bufPos As Long
    bLittleEndian As Boolean
End Type
Private encapCnt As Long
Private encaps() As tEncap

'GIOP Version may be lower than IIOP Version
Private iGIOPVersion As Integer 'Example: &H0, &H100, &H101, &H102
Private bMoreFrags As Boolean

'CopyRect();
'Private Declare Function dllCopy16B Lib "user32" Alias "CopyRect" _
'    (ByRef lpDestRect As Any, ByRef lpSourceRect As Any) As Long

'RtlMoveMemory();
Private Declare Sub dllMoveMem Lib "kernel32" Alias "RtlMoveMemory" _
    (ByRef hpvDest As Any, ByRef hpvSource As Any, ByVal cbCopy As Long)

'lstrcpyA();
Private Declare Function dllStrCpy Lib "kernel32" Alias "lstrcpyA" _
    (ByRef lpString1 As Any, ByVal lpString2 As String) As Long

Private Sub Class_Initialize()
    #If DebugMode Then
        lClassDebugID = mVBOrb.getNextClassDebugID()
        Debug.Print "'" & TypeName(Me) & "' " & lClassDebugID & " initialized"
    #End If
End Sub

Private Sub Class_Terminate()
    'Release something which VB cannot know if required
    #If DebugMode Then
        Debug.Print "'" & TypeName(Me) & "' " & CStr(lClassDebugID) & " terminated"
    #End If
End Sub

#If DebugMode Then
    Friend Property Get ClassDebugID() As Long
        ClassDebugID = lClassDebugID
    End Property
#End If

'Initialization
'To init by octets (initByOctets) do following:
' - Call oIn.initStream(oOrb, &H100, seqLen, False)
' - Call oIn.writeOctets(seqArr, seqLen)
' - Call oIn.setPos(0)
Public Sub initStream(ByVal Orb As cOrbImpl, ByVal GIOPVersion As Integer, _
    Optional ByVal lSize As Long = 1024)
    On Error GoTo ErrHandler
    If isInitialized() Then
        Call mVBOrb.VBOrb.raiseBADINVORDER(1, mVBOrb.VBOrb.CompletedNO)
    End If
    Set oOrb = Orb
    If lSize <= 0 Then lSize = 1
    ReDim buffer(0 To lSize - 1)
    bufOff = 0
    bufEof = bufOff 'Buffer is empty
    bufPos = 0
    bLittleEndian = False
    encapCnt = 0
    ReDim encaps(0 To 4)
    iGIOPVersion = GIOPVersion
    bMoreFrags = False
    Exit Sub
ErrHandler:
    Set oOrb = Nothing
    Call mVBOrb.ErrReraise("initStream")
End Sub

'Is initialized?
Public Function isInitialized() As Boolean
    isInitialized = Not oOrb Is Nothing
End Function

'Close OrbStream to reinit it next time
Public Sub destroy()
    If isInitialized() Then
        Erase buffer
        Erase encaps
        Set oOrb = Nothing
    End If
End Sub

'Prepare outgoing GIOP message
Public Sub sendGIOPPrepare()
    On Error GoTo ErrHandler
    Call writeOctet(Asc("G"))
    Call writeOctet(Asc("I"))
    Call writeOctet(Asc("O"))
    Call writeOctet(Asc("P"))
    Call writeOctet(iGIOPVersion \ &H100)  'GIOPVerMajor
    Call writeOctet(iGIOPVersion And &HFF) 'GIOPVerMinor
    Call writeSkip(6)
    Exit Sub
ErrHandler:
    Call mVBOrb.ErrReraise("sendGIOPPrepare")
End Sub

'Send outgoing GIOP message
Public Sub sendGIOPToSocket(ByVal msgType As Byte, ByVal oSock As cOrbSocket)
    On Error GoTo ErrHandler
    Call setPos(6)
    If iGIOPVersion = &H100 Then
        Call writeBoolean(bLittleEndian)
    Else
        Dim bb As Byte
        bb = 0
        If bLittleEndian Then
            bb = bb + 1
        End If
        If bMoreFrags Then
            bb = bb + 2
        End If
        Call writeOctet(bb)
    End If
    Call writeOctet(msgType)
    Call writeUlong(bufEof - 12)
    
    Call oSock.sendBuffer(buffer, bufEof)
    Exit Sub
ErrLog:
    Call oOrb.logErr("sendGIOP")
    Resume Next
ErrHandler:
    Call mVBOrb.ErrSave
    Resume ErrClose
ErrClose:
    On Error GoTo ErrLog
    'Close connection to get a new connection next time
    Call oSock.closeSocket
    On Error GoTo 0
    Call mVBOrb.ErrLoad
    Call mVBOrb.ErrReraise("sendGIOP")
End Sub

'Convert contents of an OrbStream in an IOR string
Public Sub sendToIOR(ByRef sIOR As String)
    On Error GoTo ErrHandler
    sIOR = "IOR:"
    Call setPos(0)
    For bufPos = 0 To bufEof - 1
        If buffer(bufPos) <= &HF Then
            sIOR = sIOR & "0"
        End If
        sIOR = sIOR & Hex$(buffer(bufPos))
    Next bufPos
    Exit Sub
ErrHandler:
    Call mVBOrb.ErrReraise("sendIOR")
End Sub

'Convert an IOR string in an OrbStream
Public Sub recvFromIOR(ByRef sIOR As String)
    On Error GoTo ErrHandler
    bufOff = 0
    bufEof = 0
    bufPos = 0
    Call writeEnsure((Len(sIOR) - 3) \ 2)
    'Convert string into sequence of octets
    For bufPos = 0 To bufEof - 1
        buffer(bufPos) = val("&H" & Mid$(sIOR, bufPos * 2 + 5, 2))
    Next bufPos
    bufPos = 0
    Exit Sub
ErrHandler:
    Call mVBOrb.ErrReraise("recvIOR")
End Sub

'Receive incoming GIOP message
'RET:    0 = GIOP Request, 3 = Locate Request,
Public Function recvGIOPFromSocket(ByVal oSock As cOrbSocket) As Integer
    On Error GoTo ErrHandler
    Dim recvLen As Long
    recvLen = oSock.recvCurrentLen()
    If recvLen = 0 Then
        recvGIOPFromSocket = 2005 'received len = 0 = closed
        Exit Function
    End If

    bufOff = 0
    bufEof = 0
    bufPos = 0
    Call writeEnsure(12)
    Call oSock.recvBuffer(buffer, bufPos, bufEof)
    
    If readOctet() <> Asc("G") Or readOctet() <> Asc("I") _
        Or readOctet() <> Asc("O") Or readOctet() <> Asc("P") Then
        Call mVBOrb.VBOrb.raiseMARSHAL(1, mVBOrb.VBOrb.CompletedNO, _
            "GIOP magic is wrong")
    End If
    iGIOPVersion = CInt(readOctet()) * &H100        'GIOPVerMajor
    iGIOPVersion = iGIOPVersion + CInt(readOctet()) 'GIOPVerMinor
    
    If iGIOPVersion = &H100 Then
        bLittleEndian = readBoolean()
        bMoreFrags = False
    ElseIf iGIOPVersion = &H101 Or iGIOPVersion = &H102 Then
        Dim bb As Byte
        bb = readOctet()
        bLittleEndian = (bb Mod 2 >= 1)
        bMoreFrags = (bb Mod 4 >= 2)
    Else
        Call mVBOrb.VBOrb.raiseMARSHAL(1, mVBOrb.VBOrb.CompletedNO, _
            "GIOP version " & Hex$(iGIOPVersion) & " is unsupported")
    End If
    recvGIOPFromSocket = CInt(readOctet())
    
    Dim msgSize As Long
    msgSize = readUlong()
    Call writeEnsure(msgSize)
    Call oSock.recvBuffer(buffer, bufPos, bufEof - bufPos)
    Exit Function
ErrHandler:
    Call mVBOrb.ErrReraise("recvGIOP")
End Function

Public Function getGIOPVersion() As Integer
    getGIOPVersion = iGIOPVersion
End Function

Public Property Get littleEndian() As Boolean
    littleEndian = bLittleEndian
End Property

Public Function getPos() As Long
    getPos = bufPos
End Function

Public Sub setPos(ByVal newBufPos As Long)
    On Error GoTo ErrHandler
    If newBufPos < bufOff Or newBufPos > bufEof Then
        Call mVBOrb.VBOrb.raiseMARSHAL(1, mVBOrb.VBOrb.CompletedNO, _
            "Invalid newBufPos: " & CStr(newBufPos))
    End If
    bufPos = newBufPos
    Exit Sub
ErrHandler:
    Call mVBOrb.ErrReraise("setPos")
End Sub

Public Property Get Available() As Long
    Available = bufEof - bufPos
    If Available < 0 Then Available = 0
End Property

Public Sub readSkip(ByVal skiplen As Long)
    On Error GoTo ErrHandler
    If skiplen < 0 Or bufPos + skiplen > bufEof Then
        Call mVBOrb.VBOrb.raiseMARSHAL(1, mVBOrb.VBOrb.CompletedNO, _
            "Unexpected skiplen: " & CStr(skiplen))
    End If
    bufPos = bufPos + skiplen
    Exit Sub
ErrHandler:
    Call mVBOrb.ErrReraise("readSkip")
End Sub

'To read an encapsulation
'Read sequence length first, seqLen= oIn.readUlong()
'Call readEncapOpen/readEncapClose or call oIn.readSkip(seqLen)
Public Sub readEncapOpen(ByVal seqLen As Long)
    On Error GoTo ErrHandler
    If seqLen < 0 Or bufPos + seqLen > bufEof Then
        Call mVBOrb.VBOrb.raiseMARSHAL(1, mVBOrb.VBOrb.CompletedNO, _
            "Unexpected seqlen: " & CStr(seqLen))
    End If
    If encapCnt > UBound(encaps) Then
        ReDim Preserve encaps(0 To encapCnt + 10)
    End If
    encaps(encapCnt).bufEof = bufEof
    encaps(encapCnt).bufOff = bufOff
    encaps(encapCnt).bufPos = bufPos + seqLen
    encaps(encapCnt).bLittleEndian = bLittleEndian
    encapCnt = encapCnt + 1
    bufOff = bufPos
    bufEof = bufOff + seqLen
    bLittleEndian = readBoolean()
    Exit Sub
ErrHandler:
    Call mVBOrb.ErrReraise("readEncapOpen")
End Sub

'Skip rest of encapsulation and close it
Public Sub readEncapClose()
    On Error GoTo ErrHandler
    If encapCnt = 0 Then
        Call mVBOrb.VBOrb.raiseMARSHAL(1, mVBOrb.VBOrb.CompletedNO, _
            "Unexpected EncapClose")
    End If
    encapCnt = encapCnt - 1
    bufEof = encaps(encapCnt).bufEof
    bufOff = encaps(encapCnt).bufOff
    bufPos = encaps(encapCnt).bufPos
    bLittleEndian = encaps(encapCnt).bLittleEndian
    Exit Sub
ErrHandler:
    Call mVBOrb.ErrReraise("readEncapClose")
End Sub

Public Function readOctet() As Byte
    On Error GoTo ErrHandler
    readOctet = buffer(bufPos)
    bufPos = bufPos + 1
    Exit Function
ErrHandler:
    Call mVBOrb.ErrReraise("readOctet")
End Function

'Used by sequence <octet> reader
'First ReDim seqArr!
Public Sub readOctets(ByRef seqArr() As Byte, ByVal seqLen As Long)
    On Error GoTo ErrHandler
    If seqLen < 0 Or bufPos + seqLen > bufEof Then
        Call mVBOrb.VBOrb.raiseMARSHAL(1, mVBOrb.VBOrb.CompletedNO, _
            "Unexpected seqLen: " & CStr(seqLen))
    End If
    If seqLen = 0 Then
        'Dim seqCnt As Long, seqEnd As Long
        'seqEnd = LBound(seqArr) + seqLen - 1
        'For seqCnt = LBound(seqArr) To seqEnd
        '    seqArr(seqCnt) = buffer(bufPos)
        '    bufPos = bufPos + 1
        'Next seqCnt
    Else
        If UBound(seqArr) - LBound(seqArr) + 1 < seqLen Then
            Call mVBOrb.VBOrb.raiseMARSHAL(1, mVBOrb.VBOrb.CompletedNO, _
                "seqArr too small, need: " & CStr(seqLen))
        End If
        Call dllMoveMem(seqArr(LBound(seqArr)), buffer(bufPos), seqLen)
        bufPos = bufPos + seqLen
    End If
    Exit Sub
ErrHandler:
    Call mVBOrb.ErrReraise("readOctets")
End Sub

'sequence <octet>
Public Function readSeqOctet(ByRef seqArr() As Byte) As Long
    On Error GoTo ErrHandler
    Dim seqLen As Long
    seqLen = readUlong()
    If seqLen <= 0 Then
        If seqLen < 0 Then
            Call mVBOrb.VBOrb.raiseMARSHAL(1, mVBOrb.VBOrb.CompletedNO, _
                "Unexpected seqLen: " & CStr(seqLen))
        End If
        'Second argument of MidB is always 1 for the first byte
        seqArr = MidB(buffer, 1 + bufPos, seqLen)
        'bufPos = bufPos + seqLen
    Else
        ReDim seqArr(0 To seqLen - 1)
        Call readOctets(seqArr, seqLen)
    End If
    readSeqOctet = seqLen
    Exit Function
ErrHandler:
    Call mVBOrb.ErrReraise("readSeqOctet")
End Function

Public Function readBoolean() As Boolean
    On Error GoTo ErrHandler
    Dim bb As Byte
    bb = buffer(bufPos)
    bufPos = bufPos + 1

    If bb = 1 Then
        readBoolean = True
    ElseIf bb = 0 Then
        readBoolean = False
    Else
        Call mVBOrb.VBOrb.raiseMARSHAL(1, mVBOrb.VBOrb.CompletedNO, _
            "Unexpected value: " & CStr(bb) & ", bufPos= " & CStr(bufPos))
    End If
    Exit Function
ErrHandler:
    Call mVBOrb.ErrReraise("readBoolean")
End Function

Public Function readShort() As Integer
    On Error GoTo ErrHandler
    ' Alignment
    If (bufPos - bufOff) Mod 2& > 0 Then
        bufPos = bufPos + 1&
    End If
    
    If bLittleEndian Then
        If buffer(bufPos + 1&) >= &H80 Then
            readShort = (CInt(buffer(bufPos + 1&) Xor &HFF) * &H100 _
                + CInt(buffer(bufPos))) Xor &HFF00
        Else
            readShort = CInt(buffer(bufPos + 1&)) * &H100 _
                + CInt(buffer(bufPos))
        End If
    Else
        If buffer(bufPos) >= &H80 Then
            readShort = (CInt(buffer(bufPos) Xor &HFF) * &H100 _
                + CInt(buffer(bufPos + 1&))) Xor &HFF00
        Else
            readShort = CInt(buffer(bufPos)) * &H100 _
                + CInt(buffer(bufPos + 1&))
        End If
    End If
    bufPos = bufPos + 2&
    Exit Function
ErrHandler:
    Call mVBOrb.ErrReraise("readShort")
End Function

Public Function readUshort() As Integer
    readUshort = readShort()
End Function

Public Function readLong() As Long
    On Error GoTo ErrHandler
    ' Alignment
    If (bufPos - bufOff) Mod 4& > 0 Then
        bufPos = bufPos + 4& - ((bufPos - bufOff) Mod 4&)
    End If
    
    If bLittleEndian Then
        If buffer(bufPos + 3&) >= &H80 Then
            readLong = ((((CLng(buffer(bufPos + 3&) Xor &HFF) * &H100 _
                + CLng(buffer(bufPos + 2&))) * &H100 _
                + CLng(buffer(bufPos + 1&))) * &H100 _
                + CLng(buffer(bufPos)))) Xor &HFF000000
        Else
            readLong = (((CLng(buffer(bufPos + 3&)) * &H100 _
                + CLng(buffer(bufPos + 2&))) * &H100 _
                + CLng(buffer(bufPos + 1&))) * &H100 _
                + CLng(buffer(bufPos)))
        End If
    Else
        If buffer(bufPos) >= &H80 Then
            readLong = ((((CLng(buffer(bufPos) Xor &HFF) * &H100 _
                + CLng(buffer(bufPos + 1&))) * &H100 _
                + CLng(buffer(bufPos + 2&))) * &H100 _
                + CLng(buffer(bufPos + 3&)))) Xor &HFF000000
        Else
            readLong = (((CLng(buffer(bufPos)) * &H100 _
                + CLng(buffer(bufPos + 1&))) * &H100 _
                + CLng(buffer(bufPos + 2&))) * &H100 _
                + CLng(buffer(bufPos + 3&)))
        End If
    End If
    bufPos = bufPos + 4&
    Exit Function
ErrHandler:
    Call mVBOrb.ErrReraise("readLong")
End Function

Public Function readUlong() As Long
    readUlong = readLong()
End Function

'First version was written by Holger Beer
Public Function readFloat() As Single
    On Error GoTo ErrHandler
    Dim i As Long
    Dim bytes(0 To 3) As Byte
    
    'Alignment
    If (bufPos - bufOff) Mod 4& > 0 Then
        bufPos = bufPos + 4& - ((bufPos - bufOff) Mod 4&)
    End If
        
    If bLittleEndian Then
        For i = 0 To 3
            bytes(i) = buffer(bufPos + i)
        Next
    Else
        For i = 0 To 3
            bytes(i) = buffer(bufPos + 3 - i)
        Next
    End If
    Call dllMoveMem(readFloat, bytes(0), 4&)
    bufPos = bufPos + 4
    Exit Function
ErrHandler:
    Call mVBOrb.ErrReraise("readFloat")
End Function

'First version was written by Holger Beer
Public Function readDouble() As Double
    On Error GoTo ErrHandler
    Dim i As Long
    Dim bytes(0 To 7) As Byte
    
    'Alignment
    If (bufPos - bufOff) Mod 8& > 0 Then
        bufPos = bufPos + 8& - ((bufPos - bufOff) Mod 8&)
    End If
    
    If bLittleEndian Then
        For i = 0& To 7&
            bytes(i) = buffer(bufPos + i)
        Next
    Else
        For i = 0& To 7&
            bytes(i) = buffer(bufPos + 7& - i)
        Next
    End If
    Call dllMoveMem(readDouble, bytes(0), 8&)
    bufPos = bufPos + 8&
    Exit Function
ErrHandler:
    Call mVBOrb.ErrReraise("readDouble")
End Function

Public Function readString() As String
    On Error GoTo ErrHandler
    Dim strlen As Long
    strlen = readUlong()
    If strlen <= 0 Then
        Call mVBOrb.VBOrb.raiseMARSHAL(1, mVBOrb.VBOrb.CompletedNO, _
            "Unexpected string value: strlen= " & CStr(strlen))
    End If
    'Second argument of MidB is always 1 for the first byte
    readString = StrConv(MidB(buffer, 1 + bufPos, strlen - 1), vbUnicode)
    'buffer(bufpos + strlen -1) is always equals Chr$(0)
    bufPos = bufPos + strlen
    Exit Function
ErrHandler:
    Call mVBOrb.ErrReraise("readString")
End Function

Public Function readObjRef() As cOrbObjRef
    On Error GoTo ErrHandler
    Set readObjRef = New cOrbObjRef
    If readObjRef.initByIOR(oOrb, Me) Then
        Set readObjRef = Nothing
    End If
    Exit Function
ErrHandler:
    Set readObjRef = Nothing
    Call mVBOrb.ErrReraise("readObjRef")
End Function

'Alignment only
Public Sub align(ByVal algnVal As Long)
    If (bufPos - bufOff) Mod algnVal > 0 Then
        bufPos = bufPos + algnVal - ((bufPos - bufOff) Mod algnVal)
    End If
End Sub

Public Sub writeEnsure(ByVal minSize As Long)
    If bufPos + minSize <= bufEof Then
        Exit Sub
    End If
    bufEof = bufPos + minSize
    If bufEof <= UBound(buffer) + 1 Then
        Exit Sub
    End If
    If minSize < 512 Then
        minSize = 512
    End If
    minSize = UBound(buffer) + 2 * minSize
    If bufPos = 0 Then
        ReDim buffer(0 To minSize)
    Else
        ReDim Preserve buffer(0 To minSize)
    End If
End Sub

Public Sub writeSkip(ByVal skiplen As Long)
    Call writeEnsure(skiplen)
    bufPos = bufPos + skiplen
End Sub

Public Sub writeEncapOpen(ByVal newLittleEndian As Boolean)
    On Error GoTo ErrHandler
    Call writeLong(0) 'Will be overwritten by writeEncapClose()
    If encapCnt > UBound(encaps) Then
        ReDim Preserve encaps(0 To encapCnt + 10)
    End If
    encaps(encapCnt).bufEof = bufEof
    encaps(encapCnt).bufOff = bufOff
    encaps(encapCnt).bufPos = bufPos
    encaps(encapCnt).bLittleEndian = bLittleEndian
    encapCnt = encapCnt + 1
    bufOff = bufPos
    bufEof = bufOff
    bLittleEndian = newLittleEndian
    Call writeBoolean(bLittleEndian)
    Exit Sub
ErrHandler:
    Call mVBOrb.ErrReraise("writeEncapOpen")
End Sub

Public Sub writeEncapClose()
    On Error GoTo ErrHandler
    If encapCnt = 0 Then
        Call mVBOrb.VBOrb.raiseMARSHAL(1, mVBOrb.VBOrb.CompletedNO, _
            "Unexpected EncapClose")
    End If
    encapCnt = encapCnt - 1
    Dim seqLen As Long
    seqLen = bufEof - bufOff
    bufEof = encaps(encapCnt).bufEof + seqLen
    bufOff = encaps(encapCnt).bufOff
    bufPos = encaps(encapCnt).bufPos - 4
    bLittleEndian = encaps(encapCnt).bLittleEndian
    Call writeLong(seqLen)
    Call writeSkip(seqLen)
    Exit Sub
ErrHandler:
    Call mVBOrb.ErrReraise("writeEncapClose")
End Sub

Public Sub writeOctet(ByVal wVal As Byte)
    On Error GoTo ErrHandler
    Call writeEnsure(1)
    buffer(bufPos) = wVal
    bufPos = bufPos + 1
    Exit Sub
ErrHandler:
    Call mVBOrb.ErrReraise("writeOctet")
End Sub

'Used by sequence <octet> writer
Public Sub writeOctets(ByRef seqArr() As Byte, ByVal seqLen As Long)
    On Error GoTo ErrHandler
    Call writeEnsure(seqLen)
    If seqLen <= 0 Then
        If seqLen < 0 Then
            Call mVBOrb.VBOrb.raiseMARSHAL(1, mVBOrb.VBOrb.CompletedNO, _
                "Unexpected seqLen: " & CStr(seqLen))
        End If
        'Dim seqCnt As Long, seqEnd As Long
        'seqEnd = LBound(seqArr) + seqLen - 1
        'For seqCnt = LBound(seqArr) To seqEnd
        '    buffer(bufPos) = seqArr(seqCnt)
        '    bufPos = bufPos + 1
        'Next seqCnt
    Else
        If UBound(seqArr) - LBound(seqArr) + 1 < seqLen Then
            Call mVBOrb.VBOrb.raiseMARSHAL(1, mVBOrb.VBOrb.CompletedNO, _
                "seqArr too small, need: " & CStr(seqLen))
        End If
        Call dllMoveMem(buffer(bufPos), seqArr(LBound(seqArr)), seqLen)
        bufPos = bufPos + seqLen
    End If
    Exit Sub
ErrHandler:
    Call mVBOrb.ErrReraise("writeOctets")
End Sub

'sequence <octet>
Public Sub writeSeqOctet(ByRef seqArr() As Byte, ByVal seqLen As Long)
    On Error GoTo ErrHandler
    Call writeUlong(seqLen)
    Call writeOctets(seqArr, seqLen)
    Exit Sub
ErrHandler:
    Call mVBOrb.ErrReraise("writeSeqOctet")
End Sub

Public Sub writeBoolean(ByVal wVal As Boolean)
    On Error GoTo ErrHandler
    Call writeEnsure(1)
    If wVal Then
        buffer(bufPos) = 1
    Else
        buffer(bufPos) = 0
    End If
    bufPos = bufPos + 1
    Exit Sub
ErrHandler:
    Call mVBOrb.ErrReraise("writeBoolean")
End Sub

Public Sub writeShort(ByVal wVal As Integer)
    On Error GoTo ErrHandler
    ' Alignment
    If (bufPos - bufOff) Mod 2& > 0 Then
        bufPos = bufPos + 1&
    End If
    Call writeEnsure(2&)
    
    If wVal >= 0 Then
        If bLittleEndian Then
            buffer(bufPos) = wVal And &HFF 'Mod &H100
            buffer(bufPos + 1&) = wVal \ &H100 'And &HFF not necessary
        Else
            buffer(bufPos + 1&) = wVal And &HFF 'Mod &H100
            buffer(bufPos) = wVal \ &H100 'And &HFF not necessary
            'No And is also a workaround for a 'division by zero' compiler bug
        End If
    Else
        If bLittleEndian Then
            buffer(bufPos) = wVal And &HFF 'Mod &H100
            wVal = ((wVal Xor &HFFFF) \ &H100) Xor &HFF
            buffer(bufPos + 1) = wVal And &HFF 'Mod &H100
        Else
            buffer(bufPos + 1) = wVal And &HFF 'Mod &H100
            wVal = ((wVal Xor &HFFFF) \ &H100) Xor &HFF
            buffer(bufPos) = wVal And &HFF 'Mod &H100
        End If
    End If
    bufPos = bufPos + 2
    Exit Sub
ErrHandler:
    Call mVBOrb.ErrReraise("writeShort")
End Sub

Public Sub writeUshort(ByVal wVal As Integer)
    Call writeShort(wVal)
End Sub

Public Sub writeLong(ByVal wVal As Long)
    On Error GoTo ErrHandler
    ' Alignment
    If (bufPos - bufOff) Mod 4& > 0 Then
        bufPos = bufPos + 4& - ((bufPos - bufOff) Mod 4&)
    End If
    Call writeEnsure(4&)
    
    If wVal >= 0 Then
        If bLittleEndian Then
            buffer(bufPos) = wVal And &HFF 'Mod &H100
            buffer(bufPos + 1) = (wVal \ &H100&) And &HFF 'Mod &H100
            buffer(bufPos + 2) = (wVal \ &H10000) And &HFF  'Mod &H100
            buffer(bufPos + 3) = wVal \ &H1000000 'And &HFF not necessary
        Else
            buffer(bufPos + 3) = wVal And &HFF 'Mod &H100
            buffer(bufPos + 2) = (wVal \ &H100&) And &HFF  'Mod &H100
            buffer(bufPos + 1) = (wVal \ &H10000) And &HFF 'Mod &H100
            buffer(bufPos) = wVal \ &H1000000 'And &HFF not necessary
        End If
    Else
        If bLittleEndian Then
            buffer(bufPos) = wVal And &HFF 'Mod &H100
            wVal = ((wVal Xor &HFFFFFFFF) \ &H100&) Xor &HFFFFFF
            buffer(bufPos + 1) = wVal And &HFF 'Mod &H100
            buffer(bufPos + 2) = (wVal \ &H100&) And &HFF 'Mod &H100
            buffer(bufPos + 3) = (wVal \ &H10000) And &HFF  'Mod &H100
        Else
            buffer(bufPos + 3) = wVal And &HFF 'Mod &H100
            wVal = ((wVal Xor &HFFFFFFFF) \ &H100&) Xor &HFFFFFF
            buffer(bufPos + 2) = wVal And &HFF 'Mod &H100
            buffer(bufPos + 1) = (wVal \ &H100&) And &HFF 'Mod &H100
            buffer(bufPos) = (wVal \ &H10000) And &HFF 'Mod &H100
        End If
    End If
    bufPos = bufPos + 4
    Exit Sub
ErrHandler:
    Call mVBOrb.ErrReraise("writeLong")
End Sub

Public Sub writeUlong(ByVal wVal As Long)
    Call writeLong(wVal)
End Sub

'First version was written by Holger Beer
Public Sub writeFloat(ByVal wVal As Single)
    On Error GoTo ErrHandler
    Dim i As Long
    Dim bytes(0 To 3) As Byte
    
    'Alignment
    If (bufPos - bufOff) Mod 4& > 0 Then
        bufPos = bufPos + 4& - ((bufPos - bufOff) Mod 4&)
    End If
    Call writeEnsure(4&)
    
    Call dllMoveMem(bytes(0), wVal, 4&)
    If bLittleEndian Then
        For i = 0 To 3
            buffer(bufPos + i) = bytes(i)
        Next
    Else
        For i = 0 To 3
            buffer(bufPos + 3 - i) = bytes(i)
        Next
    End If
    
    bufPos = bufPos + 4
    Exit Sub
ErrHandler:
    Call mVBOrb.ErrReraise("writeFloat")
End Sub

'First version was written by Holger Beer
Public Sub writeDouble(ByVal wVal As Double)
    On Error GoTo ErrHandler
    Dim i As Long
    Dim bytes(0 To 7) As Byte
    
    'Alignment
    If (bufPos - bufOff) Mod 8& > 0 Then
        bufPos = bufPos + 8& - ((bufPos - bufOff) Mod 8&)
    End If
    Call writeEnsure(8&)
    
    Call dllMoveMem(bytes(0), wVal, 8&)
    If bLittleEndian Then
        For i = 0 To 7
            buffer(bufPos + i) = bytes(i)
        Next
    Else
        For i = 0 To 7
            buffer(bufPos + 7 - i) = bytes(i)
        Next
    End If
    
    bufPos = bufPos + 8
    Exit Sub
ErrHandler:
    Call mVBOrb.ErrReraise("writeDouble")
End Sub

Public Sub writeString(ByRef wVal As String)
    On Error GoTo ErrHandler
    Dim strlen As Long
    strlen = Len(wVal)
    Call writeUlong(strlen + 1)
    Call writeEnsure(strlen + 1)
    Call dllStrCpy(buffer(bufPos), wVal)
    'Dim i1 As Long
    'For i1 = 0 To strlen - 1
    '    buffer(bufPos + i1) = Asc(Mid$(wVal, i1 + 1, 1))
    'Next i1
    buffer(bufPos + strlen) = 0
    bufPos = bufPos + strlen + 1
    Exit Sub
ErrHandler:
    Call mVBOrb.ErrReraise("writeString")
End Sub

Public Sub writeObjRef(ByVal oObjRef As cOrbObjRef)
    On Error GoTo ErrHandler
    If oObjRef Is Nothing Then
        'string type_id;
        Call writeString("")
        'sequence <TaggedProfile> profiles;
        Call writeUlong(0)
    Else
        Call oObjRef.writeMe(Me)
    End If
    Exit Sub
ErrHandler:
    Call mVBOrb.ErrReraise("writeObjRef")
End Sub

'SeqLen = oIn.Available
Public Sub writeStream(ByVal oIn As cOrbStream, ByVal seqLen As Long)
    On Error GoTo ErrHandler
    If seqLen > 0 Then
        Dim seqArr() As Byte
        ReDim seqArr(seqLen)
        Call oIn.readOctets(seqArr, seqLen)
        Call Me.writeOctets(seqArr, seqLen)
        Erase seqArr
    End If
    Exit Sub
ErrHandler:
    Call mVBOrb.ErrReraise("writeStream")
End Sub

'Written by Mikael Gjrloff, Sweden, 2000-08-29
'Small debugging tool to have a look in the buffer of 'Me'.
'Usage:
'   Call MyOrbStream.dumpbuffer("Description...")
'Adds to the file
'   App.Path & "\buffsniff.log"
'Creates it if it does not exists
'All of this may very well be removed... ;-)
Friend Sub dumpbuffer(ByRef sText As String)
    
    Dim i As Long
    Dim sHexChar As String
    Dim sWriteBuf As String
    Dim sWriteBufString As String
    
    If bufEof > 0 Then
        For i = 0 To UBound(buffer) - 1
            sHexChar = Hex$(buffer(i))
            sWriteBufString = sWriteBufString & Chr(buffer(i))
            If Len(sHexChar) = 1 Then sHexChar = "0" & sHexChar
            sWriteBuf = sWriteBuf & sHexChar
        Next
    Else
        sWriteBuf = "Buffer empty!!!"
    End If
    
    Dim nFile As Integer
    Dim sFile As String
    
    nFile = FreeFile
    sFile = App.Path & "\buffsniff.log"
    Open sFile For Append As nFile
    Print #nFile, sText & "=" & sWriteBuf
    Print #nFile, sText & "=" & sWriteBufString
    Close nFile

End Sub

